/*****************************************************************************/
/* Example program illustrating countours in Measurement Studio 3D graph     */
/*****************************************************************************/

/*****************************************************************************/
/* Include Files                                                             */
/*****************************************************************************/
#include <cviauto.h>
#include "3DGraphCtrl.h"
#include <ansi_c.h>
#include <cvirte.h>     
#include <userint.h>
#include "3DGraphContours.h"

/*****************************************************************************/
/* Constants and Macros                                                      */
/*****************************************************************************/
#define hrChk(f)            if (hr = (f), FAILED(hr)) goto Error; else
#define PI 3.1415926535

/*****************************************************************************/
/* Global Variables                                                          */
/*****************************************************************************/
static int              gPanel;
static CAObjHandle      gSurfaceGraph;
static CAObjHandle      gContourGraph;
static int              gEditingLevels,
                        gEditingInterval,
                        gEditingAnchor,
                        gRefreshing;
static int              gColorTable[] = { 0, 0x0000FFL, 0x00FF00L, 0xFF0000L };                     

/*****************************************************************************/
/* Forward Declarations                                                      */
/*****************************************************************************/
static HRESULT Initialize(void);
static HRESULT RefreshControls(void);
static HRESULT GetContoursFromGraph(CAObjHandle graph, CAObjHandle *contours);

/*****************************************************************************/
/* Program entry-point                                                       */
/*****************************************************************************/
int main(int argc, char *argv[])
{
    HRESULT     hr;
    
    if (InitCVIRTE(0, argv, 0) == 0)
        return -1;
    if ((gPanel = LoadPanel(0, "3DGraphContours.uir", PANEL)) < 0)
        return -1;
    hrChk (Initialize());
    DisplayPanel(gPanel);
    RunUserInterface();

Error:
    if (FAILED(hr))
        CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
    CA_DiscardObjHandle(gSurfaceGraph);
    CA_DiscardObjHandle(gContourGraph);
    DiscardPanel(gPanel);
    return 0;
}

/*****************************************************************************/
/* Callback Functions                                                        */
/*****************************************************************************/
int CVICALLBACK PlotCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    HRESULT     hr;
    double      t[21], zData[21][21];
    VARIANT     zVt, tVt;
    int         i, j;
    double      factor;
    CAObjHandle contours = 0;
    
    switch (event)
        {
        case EVENT_COMMIT:
            /* Generate data */
            if (control == PANEL_PLOT1)
                {
                for (i = 0; i < 21; ++i)
                    t[i] = (i / 20.0 - 0.5) * 2 * PI;
                for (i = 0; i < 21; ++i)
                    for(j = 0; j < 21; ++j)
                        zData[j][i] = sinh(t[i]) * cos(t[j]);
                }
            else if (control == PANEL_PLOT2)
                {
                for (i = 0; i < 21; ++i)
                    t[i] = (i / 20.0 - 0.5) * 2;
                for (i = 0; i < 21; ++i)
                    for(j = 0; j < 21; ++j)
                        zData[j][i] = -(t[i] * t[i] + pow(pow(t[j], 2.0), 1.0 / 3.0));
                }
            else
                {
                for (i = 0; i < 21; ++i)
                    t[i] = (i / 20.0 - 0.5) * 8;
                for (i = 0; i < 21; ++i)
                    for(j = 0; j < 21; ++j)
                        zData[j][i] = 8 * exp(-(t[i] * t[i] + t[j] * t[j]) / 4);
                }
                
            hrChk (CA_VariantSet1DArray(&tVt, CAVT_DOUBLE, 21, t));
            hrChk (CA_VariantSet2DArray(&zVt, CAVT_DOUBLE, 21, 21, zData));
            
            /* Display data */
            hrChk (CW3DGraphLib__DCWGraph3DPlot3DSurface(gSurfaceGraph, NULL, tVt, tVt, 
                zVt, CA_DEFAULT_VAL));
            hrChk (CW3DGraphLib__DCWGraph3DPlot3DSurface(gContourGraph, NULL, tVt, tVt, 
                zVt, CA_DEFAULT_VAL));

            hrChk (GetContoursFromGraph(gSurfaceGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetLevels(contours, NULL, 5));
            CA_DiscardObjHandle(contours);
            contours = 0;

            hrChk (GetContoursFromGraph(gContourGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetLevels(contours, NULL, 5));
                
            hrChk (RefreshControls());

Error:
            if (FAILED(hr))
                CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
            CA_VariantClear(&tVt);
            CA_VariantClear(&zVt);
            CA_DiscardObjHandle(contours);
            break;
        }
    return 0;
}

int CVICALLBACK LevelsCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    int         enabled;
    
    switch (event)
        {
        case EVENT_COMMIT:
            GetCtrlVal(gPanel, PANEL_LEVELS, &enabled);
            SetCtrlAttribute(gPanel, PANEL_LEVELS_NUM, ATTR_DIMMED, !enabled);

            SetCtrlVal(gPanel, PANEL_INTERVAL, !enabled);
            SetCtrlAttribute(gPanel, PANEL_INTERVAL_NUM, ATTR_DIMMED, enabled);
            SetCtrlAttribute(gPanel, PANEL_ANCHOR, ATTR_DIMMED, enabled);
            SetCtrlAttribute(gPanel, PANEL_ANCHOR_NUM, ATTR_DIMMED, enabled);
            
            LevelsNumCallback(gPanel, PANEL_LEVELS_NUM, EVENT_COMMIT, 0, 0, 0);
            break;
        }
    return 0;
}

int CVICALLBACK IntervalCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    int         enabled, anchor;
    
    switch (event)
        {
        case EVENT_COMMIT:
            GetCtrlVal(gPanel, PANEL_INTERVAL, &enabled);
            SetCtrlAttribute(gPanel, PANEL_INTERVAL_NUM, ATTR_DIMMED, !enabled);
            
            SetCtrlVal(gPanel, PANEL_LEVELS, !enabled);
            SetCtrlAttribute(gPanel, PANEL_ANCHOR, ATTR_DIMMED, !enabled);
            GetCtrlVal(gPanel, PANEL_ANCHOR, &anchor);
            SetCtrlAttribute(gPanel, PANEL_ANCHOR_NUM, ATTR_DIMMED, !anchor);
            SetCtrlAttribute(gPanel, PANEL_LEVELS_NUM, ATTR_DIMMED, enabled);

            IntervalNumCallback(gPanel, PANEL_INTERVAL_NUM, EVENT_COMMIT, 0, 0, 0);
            break;
        }
    return 0;
}

int CVICALLBACK LevelsNumCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    HRESULT     hr;
    int         levels;
    CAObjHandle contours = 0;
    
    switch (event)
        {
        case EVENT_COMMIT:
            if (!gRefreshing)
                {
                GetCtrlVal(gPanel, PANEL_LEVELS_NUM, &levels);
                
                hrChk (GetContoursFromGraph(gSurfaceGraph, &contours));
                hrChk (CW3DGraphLib_CWContoursSetLevels(contours, NULL, levels));
                CA_DiscardObjHandle(contours);
                contours = 0;
                
                hrChk (GetContoursFromGraph(gContourGraph, &contours));
                hrChk (CW3DGraphLib_CWContoursSetLevels(contours, NULL, levels));

                gEditingLevels = 1;
                hrChk (RefreshControls());
                gEditingLevels = 0;
                }

Error:
            if (FAILED(hr))
                CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
            CA_DiscardObjHandle(contours);
            break;
        }
    return 0;
}

int CVICALLBACK IntervalNumCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    HRESULT     hr;
    double      interval;
    CAObjHandle contours = 0;
    
    switch (event)
        {
        case EVENT_COMMIT:
            if (!gRefreshing)
                {
                GetCtrlVal(gPanel, PANEL_INTERVAL_NUM, &interval);
                
                hrChk (GetContoursFromGraph(gSurfaceGraph, &contours));
                hrChk (CW3DGraphLib_CWContoursSetInterval(contours, NULL, interval));
                CA_DiscardObjHandle(contours);
                contours = 0;

                hrChk (GetContoursFromGraph(gContourGraph, &contours));
                hrChk (CW3DGraphLib_CWContoursSetInterval(contours, NULL, interval));
                
                gEditingInterval = 1;
                hrChk (RefreshControls());
                gEditingInterval = 0;
                }

Error:
            if (FAILED(hr))
                CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
            CA_DiscardObjHandle(contours);
            break;
        }
    return 0;
}

int CVICALLBACK AnchorCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    HRESULT     hr;
    int         enabled;
    CAObjHandle contours = 0;
    
    switch (event)
        {
        case EVENT_COMMIT:
            GetCtrlVal(gPanel, PANEL_ANCHOR, &enabled);
            
            hrChk (GetContoursFromGraph(gSurfaceGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetAnchorEnabled(contours, NULL, 
                enabled ? VTRUE : VFALSE));
            CA_DiscardObjHandle(contours);
            contours = 0;
            
            hrChk (GetContoursFromGraph(gContourGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetAnchorEnabled(contours, NULL, 
                enabled ? VTRUE : VFALSE));

            SetCtrlAttribute(gPanel, PANEL_ANCHOR_NUM, ATTR_DIMMED, !enabled);

Error:
            if (FAILED(hr))
                CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
            CA_DiscardObjHandle(contours);
            break;
        }
    return 0;
}

int CVICALLBACK AnchorNumCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    HRESULT     hr;
    double      anchor;
    CAObjHandle contours = 0;
    
    switch (event)
        {
        case EVENT_COMMIT:
            if (!gRefreshing)
                {
                GetCtrlVal(gPanel, PANEL_ANCHOR_NUM, &anchor);
                
                hrChk (GetContoursFromGraph(gSurfaceGraph, &contours));
                hrChk (CW3DGraphLib_CWContoursSetAnchor(contours, NULL, anchor));
                CA_DiscardObjHandle(contours);
                contours = 0;
                
                hrChk (GetContoursFromGraph(gContourGraph, &contours));
                hrChk (CW3DGraphLib_CWContoursSetAnchor(contours, NULL, anchor));
                
                gEditingAnchor = 1;
                hrChk (RefreshControls());
                gEditingAnchor = 0;
                }

Error:
            if (FAILED(hr))
                CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
            CA_DiscardObjHandle(contours);
            break;
        }
    return 0;
}

int CVICALLBACK ShowLabelsCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    HRESULT     hr;
    int         visible;
    CAObjHandle contours = 0;
    
    switch (event)
        {
        case EVENT_COMMIT:
            GetCtrlVal(gPanel, PANEL_SHOW_LABELS, &visible);

            hrChk (GetContoursFromGraph(gSurfaceGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetLabelVisible(contours, NULL, 
                visible ? VTRUE : VFALSE));
            CA_DiscardObjHandle(contours);
            contours = 0;
            
            hrChk (GetContoursFromGraph(gContourGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetLabelVisible(contours, NULL, 
                visible ? VTRUE : VFALSE));

Error:
            if (FAILED(hr))
                CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
            CA_DiscardObjHandle(contours);
            break;
        }
    return 0;
}

int CVICALLBACK LabelFormatCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    HRESULT     hr;
    int         length;
    char        *labelFormat = 0;
    CAObjHandle contours = 0;
    
    switch (event)
        {
        case EVENT_COMMIT:
            GetCtrlAttribute (gPanel, PANEL_LABEL_FORMAT, 
                ATTR_STRING_TEXT_LENGTH, &length);
            labelFormat = malloc ((length + 1) * sizeof (char));
            if (labelFormat == NULL)
                hrChk (E_OUTOFMEMORY);
            else
                {
                GetCtrlVal(gPanel, PANEL_LABEL_FORMAT, labelFormat);
                hrChk (GetContoursFromGraph(gSurfaceGraph, &contours));
                hrChk (CW3DGraphLib_CWContoursSetLabelFormat(contours, NULL, labelFormat));
                CA_DiscardObjHandle(contours);
                contours = 0;
                hrChk (GetContoursFromGraph(gContourGraph, &contours));
                hrChk (CW3DGraphLib_CWContoursSetLabelFormat(contours, NULL, labelFormat));
                hrChk (RefreshControls());
                }

Error:
            if (FAILED(hr))
                CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
            CA_DiscardObjHandle(contours);
            free(labelFormat);
            break;
        }
    return 0;
}

int CVICALLBACK LineStyleCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    HRESULT         hr;
    int             style;
    CAObjHandle     contours = 0;
    
    switch (event)
        {
        case EVENT_COMMIT:
            GetCtrlVal(gPanel, PANEL_LINE_STYLE, &style);

            hrChk (GetContoursFromGraph(gSurfaceGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetLineStyle(contours, NULL, style));
            CA_DiscardObjHandle(contours);
            contours = 0;
            
            hrChk (GetContoursFromGraph(gContourGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetLineStyle(contours, NULL, style));
            
Error:
            if (FAILED(hr))
                CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
            CA_DiscardObjHandle(contours);
            break;
        }
    return 0;
}

int CVICALLBACK LineColorCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    HRESULT         hr;
    int             index;
    CAObjHandle     contours = 0;
    
    switch (event)
        {
        case EVENT_COMMIT:
            GetCtrlIndex(gPanel, PANEL_LINE_COLOR, &index);
            
            hrChk (GetContoursFromGraph(gSurfaceGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetLineColor(contours, NULL, gColorTable[index]));
            CA_DiscardObjHandle(contours);
            contours = 0;
            
            hrChk (GetContoursFromGraph(gContourGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetLineColor(contours, NULL, gColorTable[index]));
            
Error:
            if (FAILED(hr))
                CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
            CA_DiscardObjHandle(contours);
            break;
        }
    return 0;
}

int CVICALLBACK LabelColorCallback(int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    HRESULT         hr;
    int             index;
    CAObjHandle     contours = 0;
    
    switch (event)
        {
        case EVENT_COMMIT:
            GetCtrlIndex(gPanel, PANEL_LABEL_COLOR, &index);
            
            hrChk (GetContoursFromGraph(gSurfaceGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetLabelColor(contours, NULL, gColorTable[index]));
            CA_DiscardObjHandle(contours);
            contours = 0;
            
            hrChk (GetContoursFromGraph(gContourGraph, &contours));
            hrChk (CW3DGraphLib_CWContoursSetLabelColor(contours, NULL, gColorTable[index]));

Error:
            if (FAILED(hr))
                CA_DisplayErrorInfo(0, "ActiveX Error", hr, NULL);
            CA_DiscardObjHandle(contours);
            break;
        }
    return 0;
}

int CVICALLBACK PanelCallback(int panel, int event, void *callbackData,
        int eventData1, int eventData2)
{
    switch (event)
        {
        case EVENT_CLOSE:
            QuitUserInterface(0);
            break;
        }
    return 0;
}

/*****************************************************************************/
/* Helper Functions                                                          */
/*****************************************************************************/
static HRESULT Initialize(void)
{
    HRESULT     hr;
    
    GetObjHandleFromActiveXCtrl(gPanel, PANEL_SURFACE_GRAPH, &gSurfaceGraph);
    GetObjHandleFromActiveXCtrl(gPanel, PANEL_CONTOUR_GRAPH, &gContourGraph);
    hrChk (RefreshControls());
    
Error:
    return hr;
}

static HRESULT RefreshControls(void)
{
    HRESULT     hr;
    int         levels;
    double      interval;
    double      anchor;
    CAObjHandle plots = 0;
    CAObjHandle plot = 0;
    CAObjHandle contours = 0;
    
    gRefreshing = 1;
    
    hrChk (CW3DGraphLib__DCWGraph3DGetPlots(gSurfaceGraph, NULL, &plots));
    hrChk (CW3DGraphLib_CWPlots3DItem(plots, NULL, CA_VariantInt(1), &plot));
    hrChk (CW3DGraphLib_CWPlot3DGetContours(plot, NULL, &contours));
    hrChk (CW3DGraphLib_CWContoursGetLevels(contours, NULL, &levels));
    hrChk (CW3DGraphLib_CWContoursGetInterval(contours, NULL, &interval));
    hrChk (CW3DGraphLib_CWContoursGetAnchor(contours, NULL, &anchor));
    
    if (!gEditingLevels)
        SetCtrlVal(gPanel, PANEL_LEVELS_NUM, levels);
    if (!gEditingInterval)
        SetCtrlVal(gPanel, PANEL_INTERVAL_NUM, interval);
    if (!gEditingAnchor)
        SetCtrlVal(gPanel, PANEL_ANCHOR_NUM, anchor);
    
    gRefreshing = 0;

Error:
    CA_DiscardObjHandle(contours);
    CA_DiscardObjHandle(plot);
    CA_DiscardObjHandle(plots);
    return hr;
}

static HRESULT GetContoursFromGraph(CAObjHandle graph, CAObjHandle *contours)
{
    HRESULT         hr;
    CAObjHandle     plots = 0, plot = 0;
    
    hrChk (CW3DGraphLib__DCWGraph3DGetPlots(graph, NULL, &plots));
    hrChk (CW3DGraphLib_CWPlots3DItem(plots, NULL, CA_VariantInt(1), &plot));
    hrChk (CW3DGraphLib_CWPlot3DGetContours(plot, NULL, contours));

Error:
    CA_DiscardObjHandle(plot);
    CA_DiscardObjHandle(plots);
    return hr;
}
/*****************************************************************************/
